<?php
namespace App\Controllers;

use PDO;

class AccountController
{
    private PDO $pdo;

    public function __construct(PDO $pdo)
    {
        $this->pdo = $pdo;
    }

    public function index(): void
    {
        // Get all accounts with bank info
        $sql = "SELECT a.*, b.name as bank_name 
                FROM accounts a 
                JOIN banks b ON a.bank_id = b.id 
                ORDER BY b.name, a.account_number";

        $accounts = $this->pdo->query($sql)->fetchAll(PDO::FETCH_ASSOC);

        // Calculate total balance
        $totalBalance = array_sum(array_column($accounts, 'balance'));

        // Calculate Monthly Income & Expense
        $currentMonth = date('Y-m');
        $incomeStmt = $this->pdo->prepare("
            SELECT SUM(amount) as total 
            FROM transactions 
            WHERE transaction_type = 'credit' 
            AND status = 'success'
            AND DATE_FORMAT(transaction_date, '%Y-%m') = ?
        ");
        $incomeStmt->execute([$currentMonth]);
        $monthlyIncome = $incomeStmt->fetch(PDO::FETCH_ASSOC)['total'] ?? 0;

        $expenseStmt = $this->pdo->prepare("
            SELECT SUM(amount) as total 
            FROM transactions 
            WHERE transaction_type = 'debit' 
            AND status = 'success'
            AND DATE_FORMAT(transaction_date, '%Y-%m') = ?
        ");
        $expenseStmt->execute([$currentMonth]);
        $monthlyExpense = $expenseStmt->fetch(PDO::FETCH_ASSOC)['total'] ?? 0;

        // Get Recent Transactions (Global)
        $recentStmt = $this->pdo->query("
            SELECT t.*, a.account_name, b.name as bank_name 
            FROM transactions t
            JOIN accounts a ON t.account_id = a.id
            JOIN banks b ON a.bank_id = b.id
            ORDER BY t.transaction_date DESC
            LIMIT 5
        ");
        $recentTransactions = $recentStmt->fetchAll(PDO::FETCH_ASSOC);

        include __DIR__ . '/../../views/banking/index.php';
    }

    public function transfer(): void
    {
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            $fromAccountId = $_POST['from_account'];
            $toAccountId = $_POST['to_account'];
            $amount = floatval($_POST['amount']);
            $description = $_POST['description'] ?? 'Transfer';

            // Validation
            if ($fromAccountId == $toAccountId) {
                header("Location: index.php?page=banking/transfer&error=same_account");
                exit;
            }

            if ($amount <= 0) {
                header("Location: index.php?page=banking/transfer&error=invalid_amount");
                exit;
            }

            // Check source balance
            $stmt = $this->pdo->prepare("SELECT balance FROM accounts WHERE id = ?");
            $stmt->execute([$fromAccountId]);
            $sourceAccount = $stmt->fetch(PDO::FETCH_ASSOC);

            if (!$sourceAccount || $sourceAccount['balance'] < $amount) {
                header("Location: index.php?page=banking/transfer&error=insufficient_balance");
                exit;
            }

            // Begin transaction
            $this->pdo->beginTransaction();

            try {
                // Debit from source
                $stmt = $this->pdo->prepare("UPDATE accounts SET balance = balance - ? WHERE id = ?");
                $stmt->execute([$amount, $fromAccountId]);

                // Create debit transaction
                $stmt = $this->pdo->prepare(
                    "INSERT INTO transactions (account_id, transaction_date, description, amount, transaction_type, status, user_id) 
                     VALUES (?, NOW(), ?, ?, 'debit', 'success', ?)"
                );
                $stmt->execute([$fromAccountId, $description, $amount, $_SESSION['user_id']]);

                // Credit to destination
                $stmt = $this->pdo->prepare("UPDATE accounts SET balance = balance + ? WHERE id = ?");
                $stmt->execute([$amount, $toAccountId]);

                // Create credit transaction
                $stmt = $this->pdo->prepare(
                    "INSERT INTO transactions (account_id, transaction_date, description, amount, transaction_type, status, user_id) 
                     VALUES (?, NOW(), ?, ?, 'credit', 'success', ?)"
                );
                $stmt->execute([$toAccountId, $description, $amount, $_SESSION['user_id']]);

                // Audit log
                $stmt = $this->pdo->prepare(
                    "INSERT INTO audit_logs (user_id, action, details, ip_address) 
                     VALUES (?, 'transfer', ?, ?)"
                );
                $stmt->execute([
                    $_SESSION['user_id'],
                    "Transfer Rp " . number_format($amount, 0) . " from account $fromAccountId to $toAccountId",
                    $_SERVER['REMOTE_ADDR']
                ]);

                $this->pdo->commit();
                header("Location: index.php?page=banking&success=transfer");
                exit;

            } catch (\Exception $e) {
                $this->pdo->rollBack();
                header("Location: index.php?page=banking/transfer&error=db_error");
                exit;
            }
        }

        // Get all accounts for form
        $accounts = $this->pdo->query(
            "SELECT a.id, a.account_number, a.account_name, a.balance, b.name as bank_name 
             FROM accounts a 
             JOIN banks b ON a.bank_id = b.id 
             ORDER BY b.name, a.account_number"
        )->fetchAll(PDO::FETCH_ASSOC);

        include __DIR__ . '/../../views/banking/transfer.php';
    }

    public function deposit(): void
    {
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            $accountId = $_POST['account_id'];
            $amount = floatval($_POST['amount']);
            $description = $_POST['description'] ?? 'Setoran Tunai';
            $categoryId = !empty($_POST['category_id']) ? $_POST['category_id'] : null;

            if ($amount <= 0) {
                header("Location: index.php?page=banking/deposit&error=invalid_amount");
                exit;
            }

            $this->pdo->beginTransaction();

            try {
                // Update balance
                $stmt = $this->pdo->prepare("UPDATE accounts SET balance = balance + ? WHERE id = ?");
                $stmt->execute([$amount, $accountId]);

                // Create transaction
                $stmt = $this->pdo->prepare(
                    "INSERT INTO transactions (account_id, transaction_date, description, amount, transaction_type, category_id, status, user_id) 
                     VALUES (?, NOW(), ?, ?, 'credit', ?, 'success', ?)"
                );
                $stmt->execute([$accountId, $description, $amount, $categoryId, $_SESSION['user_id']]);

                // Audit log
                $stmt = $this->pdo->prepare(
                    "INSERT INTO audit_logs (user_id, action, details, ip_address) 
                     VALUES (?, 'deposit', ?, ?)"
                );
                $stmt->execute([
                    $_SESSION['user_id'],
                    "Deposit Rp " . number_format($amount, 0) . " to account $accountId",
                    $_SERVER['REMOTE_ADDR']
                ]);

                $this->pdo->commit();
                header("Location: index.php?page=banking&success=deposit");
                exit;

            } catch (\Exception $e) {
                $this->pdo->rollBack();
                header("Location: index.php?page=banking/deposit&error=db_error");
                exit;
            }
        }

        // Get accounts and categories
        $accounts = $this->pdo->query(
            "SELECT a.id, a.account_number, a.account_name, b.name as bank_name 
             FROM accounts a 
             JOIN banks b ON a.bank_id = b.id 
             ORDER BY b.name"
        )->fetchAll(PDO::FETCH_ASSOC);

        $categories = $this->pdo->query("SELECT * FROM categories ORDER BY name")->fetchAll(PDO::FETCH_ASSOC);

        include __DIR__ . '/../../views/banking/deposit.php';
    }

    public function withdraw(): void
    {
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            $accountId = $_POST['account_id'];
            $amount = floatval($_POST['amount']);
            $description = $_POST['description'] ?? 'Penarikan Tunai';
            $categoryId = !empty($_POST['category_id']) ? $_POST['category_id'] : null;

            if ($amount <= 0) {
                header("Location: index.php?page=banking/withdraw&error=invalid_amount");
                exit;
            }

            // Check balance
            $stmt = $this->pdo->prepare("SELECT balance FROM accounts WHERE id = ?");
            $stmt->execute([$accountId]);
            $account = $stmt->fetch(PDO::FETCH_ASSOC);

            if (!$account || $account['balance'] < $amount) {
                header("Location: index.php?page=banking/withdraw&error=insufficient_balance");
                exit;
            }

            $this->pdo->beginTransaction();

            try {
                // Update balance
                $stmt = $this->pdo->prepare("UPDATE accounts SET balance = balance - ? WHERE id = ?");
                $stmt->execute([$amount, $accountId]);

                // Create transaction
                $stmt = $this->pdo->prepare(
                    "INSERT INTO transactions (account_id, transaction_date, description, amount, transaction_type, category_id, status, user_id) 
                     VALUES (?, NOW(), ?, ?, 'debit', ?, 'success', ?)"
                );
                $stmt->execute([$accountId, $description, $amount, $categoryId, $_SESSION['user_id']]);

                // Audit log
                $stmt = $this->pdo->prepare(
                    "INSERT INTO audit_logs (user_id, action, details, ip_address) 
                     VALUES (?, 'withdraw', ?, ?)"
                );
                $stmt->execute([
                    $_SESSION['user_id'],
                    "Withdraw Rp " . number_format($amount, 0) . " from account $accountId",
                    $_SERVER['REMOTE_ADDR']
                ]);

                $this->pdo->commit();
                header("Location: index.php?page=banking&success=withdraw");
                exit;

            } catch (\Exception $e) {
                $this->pdo->rollBack();
                header("Location: index.php?page=banking/withdraw&error=db_error");
                exit;
            }
        }

        // Get accounts and categories
        $accounts = $this->pdo->query(
            "SELECT a.id, a.account_number, a.account_name, a.balance, b.name as bank_name 
             FROM accounts a 
             JOIN banks b ON a.bank_id = b.id 
             ORDER BY b.name"
        )->fetchAll(PDO::FETCH_ASSOC);

        $categories = $this->pdo->query("SELECT * FROM categories ORDER BY name")->fetchAll(PDO::FETCH_ASSOC);

        include __DIR__ . '/../../views/banking/withdraw.php';
    }
}
?>