<?php
namespace App\Controllers;

use PDO;

class AnalyticsController
{
    private PDO $pdo;

    public function __construct(PDO $pdo)
    {
        $this->pdo = $pdo;
    }

    public function index(): void
    {
        if (!isset($_SESSION['user_id'])) {
            header('Location: index.php?page=login');
            exit;
        }

        // 1. KPI Cards Data
        $totalProfit = $this->calculateTotalProfit();
        $totalVolume = $this->calculateTotalVolume();
        $avgTransaction = $this->calculateAvgTransaction();
        $successRate = $this->calculateSuccessRate();

        // 2. Monthly Trend (Income vs Expense) - Last 6 Months
        $monthlyTrend = $this->getMonthlyTrend();

        // 3. Category Distribution (Expensive only or All?) -> Let's do Expense Breakdown
        $expenseCategories = $this->getCategoryBreakdown('expense');

        // 4. Top Performing Services (PPOB)
        $topServices = $this->getTopServices();

        // 5. Profit per Bank (Income distribution)
        $bankProfit = $this->getBankProfitDistribution();

        // 6. Status Breakdown
        $statusBreakdown = $this->getStatusBreakdown();

        include __DIR__ . '/../../views/analytics/index.php';
    }

    // ... existing private functions ...

    private function getBankProfitDistribution()
    {
        // Balance per bank (or Total Income per bank?)
        // Let's show Total Balance per Bank as a proxy for "Strength"
        // OR Total Income (credits) per Bank
        $stmt = $this->pdo->query("SELECT b.name, SUM(amount) as total 
                                   FROM transactions t 
                                   JOIN accounts a ON t.account_id = a.id 
                                   JOIN banks b ON a.bank_id = b.id 
                                   WHERE t.transaction_type = 'credit' AND t.status = 'success'
                                   GROUP BY b.name 
                                   ORDER BY total DESC");
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    private function getStatusBreakdown()
    {
        $stmt = $this->pdo->query("SELECT status, COUNT(*) as count FROM transactions GROUP BY status");
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    private function calculateTotalProfit()
    {
        // Profit from PPOB + (Credit Transactions - Debit Transactions excluding PPOB cost)
        // Simplified: Sum of all (Credits) - Sum of all (Debits) + Profit Amount from Services

        // Actually, let's use the 'profit_amount' column for PPOB, and basic arithmetic for others.
        // For Bank Transfer: Fee is profit. 
        // Let's stick to the simple logic: (Sum Credit) - (Sum Debit of type 'expense') + (Sum Profit of PPOB)

        // Aggregated Profit from transactions table (profit_amount is populated for PPOB)
        $stmt = $this->pdo->query("SELECT SUM(profit_amount) FROM transactions WHERE status = 'success'");
        $ppobProfit = $stmt->fetchColumn() ?: 0;

        // Manual Administration fees (if tracked separately) could be added here.
        // For now, let's assume 'profit_amount' captures the main business profit.
        // If we want "Net Cash Flow", that's different. Let's show "Est. Profit".
        return $ppobProfit;
    }

    private function calculateTotalVolume()
    {
        $stmt = $this->pdo->query("SELECT COUNT(*) FROM transactions WHERE status = 'success'");
        return $stmt->fetchColumn() ?: 0;
    }

    private function calculateAvgTransaction()
    {
        $stmt = $this->pdo->query("SELECT AVG(amount) FROM transactions WHERE status = 'success'");
        return $stmt->fetchColumn() ?: 0;
    }

    private function calculateSuccessRate()
    {
        $total = $this->pdo->query("SELECT COUNT(*) FROM transactions")->fetchColumn();
        if ($total == 0)
            return 0;

        $success = $this->pdo->query("SELECT COUNT(*) FROM transactions WHERE status = 'success'")->fetchColumn();
        return round(($success / $total) * 100, 1);
    }

    private function getMonthlyTrend()
    {
        // Last 6 months
        $data = [];
        for ($i = 5; $i >= 0; $i--) {
            $monthStart = date('Y-m-01', strtotime("-$i months"));
            $monthEnd = date('Y-m-t', strtotime("-$i months"));
            $label = date('M Y', strtotime("-$i months"));

            // Income (Credits)
            $stmt = $this->pdo->prepare("SELECT SUM(amount) FROM transactions WHERE transaction_type = 'credit' AND status = 'success' AND transaction_date BETWEEN ? AND ?");
            $stmt->execute([$monthStart . ' 00:00:00', $monthEnd . ' 23:59:59']);
            $income = $stmt->fetchColumn() ?: 0;

            // Expense (Debits)
            $stmt = $this->pdo->prepare("SELECT SUM(amount) FROM transactions WHERE transaction_type = 'debit' AND status = 'success' AND transaction_date BETWEEN ? AND ?");
            $stmt->execute([$monthStart . ' 00:00:00', $monthEnd . ' 23:59:59']);
            $expense = $stmt->fetchColumn() ?: 0;

            $data['labels'][] = $label;
            $data['income'][] = $income;
            $data['expense'][] = $expense;
        }
        return $data;
    }

    private function getCategoryBreakdown($type = 'expense')
    {
        // Top 5 categories by amount
        $sql = "SELECT c.name, SUM(t.amount) as total 
                FROM transactions t 
                JOIN categories c ON t.category_id = c.id 
                WHERE t.transaction_type = 'debit' AND t.status = 'success' 
                GROUP BY c.name 
                ORDER BY total DESC 
                LIMIT 5";
        $stmt = $this->pdo->query($sql);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    private function getTopServices()
    {
        // Top 5 Services by Sales Volume (Count)
        $sql = "SELECT s.name, s.code, COUNT(t.id) as sales_count, SUM(t.profit_amount) as total_profit 
                FROM transactions t 
                JOIN services s ON t.service_id = s.id 
                WHERE t.status = 'success' 
                GROUP BY s.id, s.name, s.code 
                ORDER BY sales_count DESC 
                LIMIT 5";
        $stmt = $this->pdo->query($sql);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
}
