<?php
namespace App\Controllers;

use PDO;

class BankController
{
    private PDO $pdo;

    public function __construct(PDO $pdo)
    {
        $this->pdo = $pdo;
    }

    public function index(): void
    {
        $banks = $this->pdo->query("SELECT * FROM banks ORDER BY name ASC")->fetchAll(PDO::FETCH_ASSOC);
        include __DIR__ . '/../../views/banks/index.php';
    }

    public function create(): void
    {
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            $name = $_POST['name'];
            $api_endpoint = $_POST['api_endpoint'] ?? null;
            $api_key = $_POST['api_key'] ?? null;

            $stmt = $this->pdo->prepare("INSERT INTO banks (name, api_endpoint, api_key) VALUES (?, ?, ?)");
            $stmt->execute([$name, $api_endpoint, $api_key]);

            header("Location: index.php?page=banks&success=true");
            exit;
        }
        include __DIR__ . '/../../views/banks/create.php';
    }

    public function show(): void
    {
        if (!isset($_GET['id'])) {
            header("Location: index.php?page=banks");
            exit;
        }
        $id = $_GET['id'];

        // Get Bank Details
        $stmt = $this->pdo->prepare("SELECT * FROM banks WHERE id = ?");
        $stmt->execute([$id]);
        $bank = $stmt->fetch(PDO::FETCH_ASSOC);

        if (!$bank) {
            header("Location: index.php?page=banks");
            exit;
        }

        // Get Bank Transactions (via Accounts)
        // Join with users (executor), categories.
        $sql = "SELECT t.*, u.full_name as executor_name, c.name as category_name, a.account_number, a.account_name
                FROM transactions t
                JOIN accounts a ON t.account_id = a.id
                LEFT JOIN users u ON t.user_id = u.id 
                LEFT JOIN categories c ON t.category_id = c.id
                WHERE a.bank_id = ?
                ORDER BY t.transaction_date DESC";

        $stmt = $this->pdo->prepare($sql);
        $stmt->execute([$id]);
        $transactions = $stmt->fetchAll(PDO::FETCH_ASSOC);

        include __DIR__ . '/../../views/banks/show.php';
    }

    public function edit(): void
    {
        if (!isset($_GET['id'])) {
            header("Location: index.php?page=banks");
            exit;
        }
        $id = $_GET['id'];

        $stmt = $this->pdo->prepare("SELECT * FROM banks WHERE id = ?");
        $stmt->execute([$id]);
        $bank = $stmt->fetch();

        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            $name = $_POST['name'];
            $api_endpoint = $_POST['api_endpoint'] ?? null;
            $api_key = $_POST['api_key'] ?? null;

            $stmt = $this->pdo->prepare("UPDATE banks SET name = ?, api_endpoint = ?, api_key = ? WHERE id = ?");
            $stmt->execute([$name, $api_endpoint, $api_key, $id]);

            header("Location: index.php?page=banks&success=updated");
            exit;
        }


        include __DIR__ . '/../../views/banks/edit.php';
    }

    public function import(): void
    {
        if (!isset($_GET['id'])) {
            header("Location: index.php?page=banks");
            exit;
        }
        $id = $_GET['id'];

        // Verify bank exists
        $stmt = $this->pdo->prepare("SELECT * FROM banks WHERE id = ?");
        $stmt->execute([$id]);
        $bank = $stmt->fetch(PDO::FETCH_ASSOC);

        if (!$bank) {
            header("Location: index.php?page=banks");
            exit;
        }

        if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_FILES['csv_file'])) {
            $file = $_FILES['csv_file'];

            if ($file['error'] === UPLOAD_ERR_OK) {
                $handle = fopen($file['tmp_name'], 'r');
                $imported = 0;
                $skipped = 0;

                // Skip header row
                fgetcsv($handle);

                while (($row = fgetcsv($handle)) !== false) {
                    // Expected CSV format: Account Number, Date, Description, Amount, Type (credit/debit), Category
                    if (count($row) < 5) {
                        $skipped++;
                        continue;
                    }

                    [$accountNumber, $date, $description, $amount, $type] = $row;
                    $categoryName = $row[5] ?? null;

                    // Find or create account for this bank
                    $stmt = $this->pdo->prepare("SELECT id FROM accounts WHERE bank_id = ? AND account_number = ?");
                    $stmt->execute([$id, $accountNumber]);
                    $account = $stmt->fetch();

                    $accountId = null;
                    if (!$account) {
                        // Create new account
                        $stmt = $this->pdo->prepare("INSERT INTO accounts (bank_id, account_number, account_name, balance) VALUES (?, ?, ?, 0)");
                        $stmt->execute([$id, $accountNumber, 'Imported Account']);
                        $accountId = $this->pdo->lastInsertId();
                    } else {
                        $accountId = $account['id'];
                    }

                    // Find category if provided
                    $categoryId = null;
                    if ($categoryName) {
                        $stmt = $this->pdo->prepare("SELECT id FROM categories WHERE name = ?");
                        $stmt->execute([$categoryName]);
                        $cat = $stmt->fetch();
                        if ($cat) {
                            $categoryId = $cat['id'];
                        }
                    }

                    // Insert transaction
                    $stmt = $this->pdo->prepare(
                        "INSERT INTO transactions (account_id, transaction_date, description, amount, transaction_type, category_id, status, user_id) 
                         VALUES (?, ?, ?, ?, ?, ?, 'success', ?)"
                    );
                    $stmt->execute([
                        $accountId,
                        $date,
                        $description,
                        abs(floatval($amount)),
                        strtolower($type) === 'credit' ? 'credit' : 'debit',
                        $categoryId,
                        $_SESSION['user_id']
                    ]);

                    // Update account balance
                    $balanceChange = strtolower($type) === 'credit' ? floatval($amount) : -floatval($amount);
                    $this->pdo->prepare("UPDATE accounts SET balance = balance + ? WHERE id = ?")->execute([$balanceChange, $accountId]);

                    $imported++;
                }

                fclose($handle);

                header("Location: index.php?page=banks/show&id=$id&imported=$imported&skipped=$skipped");
                exit;
            }
        }

        header("Location: index.php?page=banks/show&id=$id&error=upload");
        exit;
    }
}
?>