<?php
namespace App\Controllers;

use PDO;

class DashboardController
{
    private PDO $pdo;

    public function __construct(PDO $pdo)
    {
        $this->pdo = $pdo;
    }

    public function index(): void
    {
        $filter = $_GET['filter'] ?? 'all';
        [$startDate, $endDate] = $this->getFilterDates($filter);

        // Global Stats
        $totalIncome = $this->fetchTotal('credit', null, $startDate, $endDate);
        $totalExpense = $this->fetchTotal('debit', null, $startDate, $endDate);
        $balance = $totalIncome - $totalExpense;

        // Profit Stats (Sum of profit_amount from transactions)
        $totalProfit = $this->fetchProfit($startDate, $endDate);

        // Per-Bank Stats
        $banks = $this->pdo->query("SELECT * FROM banks")->fetchAll(PDO::FETCH_ASSOC);
        foreach ($banks as &$bank) {
            $bankIncome = $this->fetchTotal('credit', $bank['id'], $startDate, $endDate);
            $bankExpense = $this->fetchTotal('debit', $bank['id'], $startDate, $endDate);
            $bank['balance'] = $bankIncome - $bankExpense;
        }

        // Recent Transactions (Always show actual recent ones, independent of filter, or should it filter? 
        // Let's filter to be consistent with the view context)
        $recentSql = "
            SELECT t.*, b.name as bank_name, c.name as category_name 
            FROM transactions t 
            JOIN accounts a ON t.account_id = a.id 
            JOIN banks b ON a.bank_id = b.id 
            LEFT JOIN categories c ON t.category_id = c.id
            WHERE 1=1
        ";
        $recentParams = [];

        if ($startDate && $endDate) {
            $recentSql .= " AND t.transaction_date BETWEEN :start AND :end";
            $recentParams[':start'] = $startDate;
            $recentParams[':end'] = $endDate;
        }

        $recentSql .= " ORDER BY t.transaction_date DESC LIMIT 5";

        $stmt = $this->pdo->prepare($recentSql);
        $stmt->execute($recentParams);
        $recentTransactions = $stmt->fetchAll(PDO::FETCH_ASSOC);

        // Fetch data for Modal (Accounts & Categories)
        $accounts = $this->pdo->query("SELECT a.*, b.name as bank_name FROM accounts a JOIN banks b ON a.bank_id = b.id ORDER BY b.name ASC")->fetchAll(PDO::FETCH_ASSOC);
        $categories = $this->pdo->query("SELECT * FROM categories ORDER BY name ASC")->fetchAll(PDO::FETCH_ASSOC);

        include __DIR__ . '/../../views/dashboard.php';
    }

    private function getFilterDates(string $filter): array
    {
        switch ($filter) {
            case 'today':
                return [date('Y-m-d 00:00:00'), date('Y-m-d 23:59:59')];
            case 'week':
                // Monday to Sunday
                return [date('Y-m-d 00:00:00', strtotime('monday this week')), date('Y-m-d 23:59:59', strtotime('sunday this week'))];
            case 'month':
                return [date('Y-m-01 00:00:00'), date('Y-m-t 23:59:59')];
            case 'all':
            default:
                return [null, null];
        }
    }

    private function fetchTotal(string $type, ?int $bankId = null, ?string $start = null, ?string $end = null): float
    {
        $sql = "SELECT SUM(amount) FROM transactions t JOIN accounts a ON t.account_id = a.id WHERE transaction_type = :type";
        $params = [':type' => $type];

        if ($bankId) {
            $sql .= " AND a.bank_id = :bank_id";
            $params[':bank_id'] = $bankId;
        }

        if ($start && $end) {
            $sql .= " AND t.transaction_date BETWEEN :start AND :end";
            $params[':start'] = $start;
            $params[':end'] = $end;
        }

        $stmt = $this->pdo->prepare($sql);
        $stmt->execute($params);
        return (float) $stmt->fetchColumn();
    }

    private function fetchProfit(?string $start = null, ?string $end = null): float
    {
        $sql = "SELECT SUM(profit_amount) FROM transactions WHERE 1=1";
        $params = [];

        if ($start && $end) {
            $sql .= " AND transaction_date BETWEEN :start AND :end";
            $params[':start'] = $start;
            $params[':end'] = $end;
        }

        $stmt = $this->pdo->prepare($sql);
        $stmt->execute($params);
        return (float) $stmt->fetchColumn();
    }
}
?>